/**
 * Example code for interfacing with the N350 sensor with an Arduino
 * jb, 2024-09-19
 * (c) Strain Measurement Devices, Inc
 * This code is unrestricted / public domain <http://unlicense.org>
 *
 * To run this example, connect the N350 and your Arduino as follows:
 *  pin 1 (red wire)    <-> 5v
 *  pin 2 (green wire)  <-> A5 (SCL)
 *  pin 3 (yellow wire) <-> A4 (SDA)
 *  pin 4 (orange wire) <-> (do not connect)
 *  pin 5 (black wire)  <-> GND
 *
 * You will also need a 4.7k pull-up resistor for SDA and SCL
 *      (connect two resistors between those pins and 5v)
 *
 * Once uploaded, you can get the read pressure through opening
 *  a serial monitor with baud 115200
 */

#include <Wire.h>

// change the address below to match your N350
//   for example 0x78 for N350-XX78, 0x28 for N350-XX28, etc
#define N350_ADDR 0x78

// full scale pressure if your sensor
//   for example 30psi for N350-30XX
#define FULL_SCALE 30.0

void setup() {
    // Initialization
    Wire.begin();
    Serial.begin(115200);
}

void loop() {
    uint8_t stat;
    uint8_t i2cData[4];
    int32_t sensorDataI;
    float pressure;

    delay(500);

    // read from the N350
    stat = Wire.requestFrom(N350_ADDR, 4);
    if(stat != 4){
        Serial.println("No data received from N350. Check address");
        return;
    }
    for(int i=0;i<4;i++){
        i2cData[i] = Wire.read();
    }

    // Convert the bytes read into digital counts
    // The data received has the following format
    //  /-------------\  /-------------------\  /------------------\  /-----------------\
    //   Sensor Status -- Sensor Data (23:16) -- Sensor Data (15:8) -- Sensor Data (7:0)
    //  \-------------/  \-------------------/  \------------------/  \-----------------/
    sensorDataI = ((uint32_t)i2cData[1] << 16) | ((uint32_t)i2cData[2] << 8) | i2cData[3];

    // Convert the digital counts into pressure, with the equation
    //      psi = ((counts - 1677824) / 13421568) * FULL_SCALE
    // where FULL_SCALE is 30psi for N350-30XX
    pressure = (float)sensorDataI;
    pressure -= 1677824.0;
    pressure /= 13421568.0;
    pressure *= FULL_SCALE;

    // We print out the read pressure
    Serial.print("Pressure: ");
    Serial.println(pressure);
}
